/**
 * study_timer.js - background service-worker helper
 *
 * Message types from popup:
 *  {type:'studyStart'}   → start timer on current active tab (if course URL)
 *  {type:'studyStop'}    → stop timer, flush remaining minutes
 *  {type:'studyStatus'}  → return {active, elapsedMs, url}
 *
 * Internal persisted state in chrome.storage.local:
 *  {
 *    active: boolean,
 *    startTime: number,        // epoch ms of current focus start
 *    elapsedSec: number,       // seconds already POSTed (multiples of 60)
 *    url: string,              // course URL being tracked
 *    pauseStart: number        // epoch ms when pause started (null if not paused)
 *  }
 */
const STORAGE_KEY = 'studyTimer';
const ALARM_NAME = 'studyTick';
const POST_INTERVAL_MIN = 5;
const API_ENDPOINT = 'https://zsp-api.yoima.com/saveStudyTime.php';
const GET_STUDY_TIME_ENDPOINT = 'https://zsp-api.yoima.com/getStudyTime.php';

// -----------------------------------------------------------------------------
// Utility
// -----------------------------------------------------------------------------
function isCourseUrl(url) {
  return /^https:\/\/www\.nnn\.ed\.nico\/courses\/\d+\/chapters\/\d+/.test(url);
}

/**
 * courses/123/chapters/456 以降のパス要素を切り落として正規化
 * 例: https://.../chapters/456/evaluation_test/999 → https://.../chapters/456
 */
function normalizeCourseUrl(url) {
  const m = url.match(/^https:\/\/www\.nnn\.ed\.nico\/courses\/\d+\/chapters\/\d+/);
  return m ? m[0] : url;
}

async function getState() {
  const res = await chrome.storage.local.get(STORAGE_KEY);
  return res[STORAGE_KEY] || { active: false, elapsedSec: 0 };
}
async function setState(state) {
  await chrome.storage.local.set({ [STORAGE_KEY]: state });
}

// -----------------------------------------------------------------------------
// Study time fetching
// -----------------------------------------------------------------------------
/**
 * 今日の累計学習時間（分）を取得
 * @param {string} classUrl - 授業URL
 * @returns {Promise<number|null>} 学習時間（分）、エラー時はnull
 */
async function fetchTodayStudyTime(classUrl) {
  try {
    const { jwt } = await chrome.storage.local.get('jwt');
    if (!jwt) {
      console.warn('JWT token not found');
      return null;
    }

    const url = `${GET_STUDY_TIME_ENDPOINT}?class_url=${encodeURIComponent(classUrl)}`;
    const res = await fetch(url, {
      method: 'GET',
      headers: {
        'Content-Type': 'application/json',
        'Authorization': 'Bearer ' + jwt,
        'X-Auth-Token': jwt
      }
    });

    if (!res.ok) {
      console.error('fetchTodayStudyTime API error:', res.status, await res.text());
      return null;
    }

    const data = await res.json();
    return data.ok ? data.minutes : null;
  } catch (e) {
    console.error('fetchTodayStudyTime error:', e);
    return null;
  }
}

// -----------------------------------------------------------------------------
// Timer control
// -----------------------------------------------------------------------------
async function startTimer(url) {
  if (!isCourseUrl(url)) return { ok: false, error: 'Not a course URL' };

  // ログイン状態をチェック
  const { jwt } = await chrome.storage.local.get('jwt');
  if (!jwt) {
    return { 
      ok: false, 
      error: 'login_required',
      message: 'ログインしていない場合、ログインすることでタイマー機能が使えるようになります。'
    };
  }

  url = normalizeCourseUrl(url);

  // 同じ日・同じ授業URLの累計時間を取得
  const todayMinutes = await fetchTodayStudyTime(url);

  const now = Date.now();
  const state = {
    active: true,
    startTime: now,
    elapsedSec: 0,
    url,
    pauseStart: null
  };
  await setState(state);
  await chrome.alarms.create(ALARM_NAME, { periodInMinutes: 1 });
  
  // 累計時間も含めて返す
  return { 
    ok: true, 
    todayMinutes: todayMinutes || 0,
    message: todayMinutes > 0 ? `今日この授業で既に${todayMinutes}分学習済みです` : null
  };
}

async function stopTimer(opts = {}) {
  const state = await getState();
  if (state.active) {
    await flushMinutes(state);
  }
  await chrome.alarms.clear(ALARM_NAME);
  await setState({ active: false, elapsedSec: 0 });
  // ポモドーロ/45・15を一時停止（フェーズ切替など内部呼び出し時は抑制可能）
  if (!opts.suppressAdvPause) {
    try {
      await chrome.runtime.sendMessage({ type: 'advancedTimerPause', reason: opts.reason || 'study_timer_stopped' });
    } catch (e) {
      // リスナー不在時などは無視
    }
  }
  return { ok: true };
}

// -----------------------------------------------------------------------------
// Alarm handler
// -----------------------------------------------------------------------------
chrome.alarms.onAlarm.addListener(async (alarm) => {
  if (alarm.name !== ALARM_NAME) return;
  const state = await getState();
  if (!state.active) return;

  // Check current active tab URL
  const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
  const now = Date.now();

  // コース外 URL での一時停止ロジック
  if (!tab || !tab.url || !isCourseUrl(tab.url)) {
    if (!state.pauseStart) {
      // 一時停止開始
      state.pauseStart = now;
      await setState(state);
    } else if (now - state.pauseStart >= POST_INTERVAL_MIN * 60 * 1000) {
      // 5分経過 → 自動停止＆サーバ保存、通知を表示
      await stopTimer({ reason: 'auto_off_course' });
      await showTimeoutNotification();
    }
    return;
  }

  // コースに戻った → pause を解除し、一時停止時間を集中時間として追加
  if (state.pauseStart) {
    const pauseDuration = now - state.pauseStart;
    // pauseDuration（一時停止していた時間）を集中時間として追加するため、
    // startTimeを現在時刻から経過時間分だけ前に設定し直す
    const currentElapsedSec = Math.floor((now - state.startTime) / 1000);
    const totalElapsedSec = currentElapsedSec + Math.floor(pauseDuration / 1000);
    state.startTime = now - (totalElapsedSec * 1000);
    state.pauseStart = null;
    await setState(state);
  }

  const totalElapsedSec = Math.floor((now - state.startTime) / 1000);
  let unpostedSec = totalElapsedSec - state.elapsedSec;

  // URL changed?
  const normalizedTab = normalizeCourseUrl(tab.url);
  if (isCourseUrl(tab.url) && normalizedTab !== state.url) {
    // flush remaining seconds (rounded to min) for old URL
    await flushMinutes(state, true);
    // reset startTime & counters for new URL
    state.url = normalizedTab;
    state.startTime = now;
    state.elapsedSec = 0;
    await setState(state);
    return;
  }

  // Accumulate and send every 5 minutes
  if (unpostedSec >= POST_INTERVAL_MIN * 60) {
    const minutesToSend = Math.floor(unpostedSec / 60);
    const wholeChunks = Math.floor(minutesToSend / POST_INTERVAL_MIN) * POST_INTERVAL_MIN;
    if (wholeChunks > 0) {
      const ok = await postMinutes(state.url, wholeChunks);
      if (ok) {
        state.elapsedSec += wholeChunks * 60;
        await setState(state);
      }
    }
  }
});

// Flush any pending minutes (called on URL change or stop)
async function flushMinutes(state, reset = false) {
  const now = Date.now();
  const totalElapsedSec = Math.floor((now - state.startTime) / 1000);
  const unpostedSec = totalElapsedSec - state.elapsedSec;
  if (unpostedSec <= 0) return;

  const minutesToSend = Math.ceil(unpostedSec / 60);
  await postMinutes(state.url, minutesToSend);

  if (reset) {
    // leave elapsedSec=0 for new URL
  } else {
    state.elapsedSec += minutesToSend * 60;
    await setState(state);
  }
}

async function postMinutes(classUrl, minutes) {
  try {
    // get JWT token
    const { jwt } = await chrome.storage.local.get('jwt');
    if (!jwt) return false;

    const res = await fetch(API_ENDPOINT, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'Authorization': 'Bearer ' + jwt,
        'X-Auth-Token': jwt
      },
      body: JSON.stringify({ class_url: classUrl, minutes })
    });
    return res.ok;
  } catch (e) {
    console.error('postMinutes error', e);
    return false;
  }
}

// -----------------------------------------------------------------------------
// Message interface for popup
// -----------------------------------------------------------------------------
chrome.runtime.onMessage.addListener((msg, _sender, sendResponse) => {
  (async () => {
    if (msg.type === 'studyStart') {
      const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
      const url = tab?.url || '';
      const result = await startTimer(url);
      sendResponse(result);
      if (result.ok) {
        const state = await getState();
        state.currentElapsedMs = 0;
        broadcastState('timer', { mode: 'manual', ...state });
      }
    } else if (msg.type === 'studyStop') {
      const res = await stopTimer();
      sendResponse(res);
      const state = await getState();
      broadcastState('timer', { mode: 'manual', ...state });
    } else if (msg.type === 'studyStatus') {
      const state = await getState();
      if (state.active) {
        const now = Date.now();
        state.currentElapsedMs = now - state.startTime;
      }
      sendResponse(state);
      // ブロードキャストを追加
      broadcastState('timer', { mode: 'manual', ...state });
    }
  })();
  return true; // indicate async
});

// -----------------------------------------------------------------------------
// Browser shutdown handler
// -----------------------------------------------------------------------------
chrome.runtime.onSuspend.addListener(async () => {
  const state = await getState();
  if (state.active) {
    await flushMinutes(state);
  }
});

// -----------------------------------------------------------------------------
// On startup, disable any active timer
// -----------------------------------------------------------------------------
  (async () => {
    const state = await getState();
    if (state.active) {
      // 拡張機能起動時にアクティブなタイマーがある場合、残り時間を保存してタイマーを無効化
    // デバッグログを削除（静かに処理）
      
      // 残り時間をサーバーに保存
      await flushMinutes(state);
    
    // アラームをクリア
    await chrome.alarms.clear(ALARM_NAME);
    
    // タイマー状態を無効化
    await setState({ active: false, elapsedSec: 0 });
    
    // 例外的な処理のため、ユーザーには通知しない
  }
})();

// -----------------------------------------------------------------------------
// Notification
// -----------------------------------------------------------------------------
async function showTimeoutNotification() {
  try {
    await chrome.notifications.create({
      type: 'basic',
      iconUrl: 'icons/icon48.png',
      title: 'ZEN Study Plus',
      message: '5分間ZEN Studyを開いていないため、集中タイマーを自動停止しました。'
    });
  } catch (e) {
    console.error('Notification error:', e);
  }
}
